Rem
Rem $Header: rdbms/demo/rac_wrappers/prvtaqadshrd.sql /main/2 2010/04/22 23:33:33 swshekha Exp $
Rem
Rem prvtaqadshrd.sql
Rem
Rem Copyright (c) 2009, 2010, Oracle and/or its affiliates. 
Rem All rights reserved. 
Rem
Rem    NAME
Rem      prvtaqadshrd.sql - <one-line expansion of the name>
Rem
Rem    DESCRIPTION
Rem      <short description of component this file declares/defines>
Rem
Rem    NOTES
Rem      <other useful comments, qualifications, etc.>
Rem
Rem    MODIFIED   (MM/DD/YY)
Rem    swshekha    09/29/09 - Created
Rem


SET FEEDBACK 1
SET NUMWIDTH 10
SET LINESIZE 80
SET TRIMSPOOL ON
SET TAB OFF
SET PAGESIZE 100


CREATE OR REPLACE PACKAGE BODY dbms_aqadm_wrapper AS

 ---------------------------------------------------------------------------
  
  /* internal procedure to generate dbms_aq_wrapper package header */
 PROCEDURE create_dbmsaq (
                          schema       varchar2,
                          type_name    varchar2
                         ) IS
    own_arr          col_tab;
    type_arr         col_tab;

  BEGIN

    if (schema IS NOT NULL) then  /* called from create_queue_table */
      own_arr(1) := schema;
      type_arr(1) := type_name;

    else      /* schema and type are null if called from make_sharded_queue */
      SELECT distinct t.owner, t.type_name BULK COLLECT INTO own_arr, type_arr
      from all_queue_tables qt, all_types t
      where sys.dbms_assert.enquote_literal(t.owner||'.'||t.type_name)  = 
             sys.dbms_assert.enquote_literal(qt.OBJECT_TYPE);
    end if;
    maptab_access.insert_type_table(own_arr, type_arr);
    
     /* generate dbms_aq_wrapper package header */
    maptab_access.generate_dbms_aq;

    /* generate dbms_aq_wrapper package body */
    maptab_access.generate_prvt_aq;

  EXCEPTION
   when no_data_found then
     NULL;
   when others then
     RAISE;
 END create_dbmsaq;

 ---------------------------------------------------------------------------
  /* internal procedure to parse queue/queue_table name */
  PROCEDURE parse_name(
             full_name      IN      VARCHAR2,
             canon_schema   OUT     VARCHAR2,
             canon_name     OUT     VARCHAR2,
             schema_indx    IN      BOOLEAN DEFAULT TRUE) IS

    v_name             VARCHAR2(3200);
    schema             VARCHAR2(30);
    name1              VARCHAR2(30);
    name2              VARCHAR2(30);
    name3              VARCHAR2(30);
    dblink             VARCHAR2(300);
    nextpos            INTEGER;

    invld_name  EXCEPTION;
    
  BEGIN
    BEGIN
      v_name := DBMS_ASSERT.QUALIFIED_SQL_NAME(full_name);
      dbms_utility.name_tokenize(v_name,name1,name2,name3,dblink,nextpos); 
    EXCEPTION
      WHEN OTHERS THEN
      RAISE invld_name;
    END;
    IF(name3 IS NOT NULL OR dblink IS NOT NULL) THEN
      RAISE invld_name;
    END IF;

    IF(name2 IS NULL) THEN
      canon_name := DBMS_ASSERT.QUALIFIED_SQL_NAME(name1);
      schema := NULL;
    ELSE
      schema := name1;
      canon_name := DBMS_ASSERT.QUALIFIED_SQL_NAME(name2);
    END IF;

    IF(schema IS NULL AND schema_indx = TRUE) THEN
       select sys_context('userenv','current_user')
      
          into canon_schema from dual;
    ELSIF(schema IS NOT NULL) THEN
      canon_schema := DBMS_ASSERT.SCHEMA_NAME(schema);
    END IF;
   
  EXCEPTION
    WHEN invld_name THEN
     RAISE_APPLICATION_ERROR(-20051, 'invalid value ' || full_name ||
                          ', name should be of the form [SCHEMA.]NAME');
    WHEN OTHERS THEN
     RAISE;
    
  END parse_name;

 ----------------------------------------------------------------------------
  
  /* internal procedure to create storage clause . This function creates 
     storage clause for the queue table passed in the argument from 
     different views available*/
  FUNCTION crt_storage_clause(
         schema   IN VARCHAR2,
         q_table  IN VARCHAR2 ) 

   RETURN VARCHAR2 AS
 
    storage_clause    VARCHAR2(500);
    tspace_name       VARCHAR2(30);
    ini_extent        NUMBER;
    nextextent        NUMBER;
    minextent         NUMBER;
    maxextent         NUMBER;
    pct_incr          NUMBER;
    free_list         NUMBER;
    freelist_grps     NUMBER;
    buffer            VARCHAR2(7);

  BEGIN
    BEGIN
      select tablespace_name, initial_extent, next_extent, min_extents, 
          max_extents, pct_increase, freelists, freelist_groups, buffer_pool 
      into tspace_name,ini_extent, nextextent, minextent, maxextent, pct_incr,
        free_list, freelist_grps, buffer
      from all_tables
      where owner = schema and table_name = q_table; 
    EXCEPTION
      WHEN NO_DATA_FOUND THEN
      maptab_access.sel_dbatab(schema, q_table,tspace_name,ini_extent,
      nextextent,minextent,maxextent,pct_incr,free_list,freelist_grps,buffer);
      NULL;
    END;

  
    storage_clause :=' TABLESPACE ' || tspace_name || 
                   ' STORAGE (INITIAL ' || ini_extent || 
                              ' NEXT '  || nextextent ||
                              ' MINEXTENTS ' || minextent || 
                              ' MAXEXTENTS ' || maxextent || 
                              ' PCTINCREASE ' || pct_incr || 
                              ' FREELISTS ' || free_list || 
                              ' FREELIST GROUPS ' || freelist_grps || 
                              ' BUFFER_POOL ' || buffer || ' )';

    RETURN storage_clause;
 
  END crt_storage_clause;

 -----------------------------------------------------------------------------

  /* internal procedure to create sharded queues of a particular logical 
     queue */
  PROCEDURE shard_queue(
         qt_schema           IN VARCHAR2,
         qtable              IN VARCHAR2,
         logical_q           IN VARCHAR2,
         queue_type          IN VARCHAR2,
         max_retry           IN NUMBER,
         retry_delay         IN NUMBER,
         retention_time      IN VARCHAR2,
         usr_comment         IN VARCHAR2,
         map_arr             IN OUT map_table_arr,
         phy_qt_cnt          IN NUMBER) IS

 
     phy_qname       VARCHAR2(30);
     q_type          BINARY_INTEGER;
     retention       NUMBER;
     v_col           BINARY_INTEGER;

  BEGIN
    IF(queue_type = 'NORMAL_QUEUE') THEN
      q_type := dbms_aqadm.NORMAL_QUEUE;
    ELSIF (queue_type = 'EXCEPTION_QUEUE') THEN
      q_type := dbms_aqadm.EXCEPTION_QUEUE;
    END IF;

    IF(retention_time = 'FOREVER') THEN
      retention := dbms_aqadm.INFINITE;
    ELSE
      retention := TO_NUMBER(retention_time);
    END IF;

    FOR indx IN 1..phy_qt_cnt  LOOP 
      IF(map_arr(indx).physical_qt = qtable) THEN
         phy_qname := logical_q;
      ELSE
        phy_qname := NLS_UPPER('sharded$_q_' || 
                               wrapperuser.q$_name_seq.nextval);
        sys.dbms_aqadm.create_queue(
            queue_name => phy_qname,
            queue_table => map_arr(indx).physical_qt,
            queue_type => q_type,
            max_retries => max_retry,
            retry_delay => retry_delay,
            retention_time => retention,
            comment => usr_comment);
      END IF;
     
      IF (map_arr(indx).physical_queue IS NOT NULL)  THEN
        v_col := map_arr.LAST;
        map_arr(v_col + 1) :=  map_arr(indx);
        map_arr(v_col + 1).physical_queue := phy_qname;
        map_arr(v_col + 1).logical_queue := logical_q;
      ELSE 
        map_arr(indx).logical_queue := logical_q;
        map_arr(indx).physical_queue := phy_qname;
      END IF;
    END LOOP;

  EXCEPTION
    WHEN NO_DATA_FOUND THEN
      dbms_output.put_line('ERROR IN SHARD QUEUE');                          
    RAISE;

  END shard_queue;

 -----------------------------------------------------------------------------

  /* internal procedure to create subscribers for sharded queues */
  PROCEDURE replicate_subscribers(
                  schema   IN VARCHAR2, 
                  q_name   IN VARCHAR2,
                  map_arr  IN map_table_arr) IS
   
    subscriber_info  sys.aq$_agent;
    rule_info        VARCHAR2(32767);
    transformation   VARCHAR2(65);
    trans_schema     VARCHAR2(30);
    trans_name       VARCHAR2(30);
    queue_prop       BOOLEAN;
    delivery         PLS_INTEGER;

    CURSOR subs_cur IS
        SELECT consumer_name, address, protocol, transformation, rule,
               delivery_mode, queue_to_queue
        FROM all_queue_subscribers 
        WHERE  queue_name = q_name AND owner = schema; 

  BEGIN
    /* Get info for each subscriber of a queue */
    FOR subs IN subs_cur LOOP

      subscriber_info := sys.aq$_agent(subs.consumer_name, subs.address,
                                          subs.protocol);      
   
      IF (subs.delivery_mode = 'PERSISTENT') THEN
        delivery := sys.DBMS_AQADM.PERSISTENT;
      ELSIF (subs.delivery_mode = 'BUFFERED') THEN
        delivery := sys.DBMS_AQADM.BUFFERED;
      ELSE
        delivery := sys.DBMS_AQADM.PERSISTENT_OR_BUFFERED;
      END IF;
  
      IF(subs.queue_to_queue = 'TRUE') THEN
        queue_prop := TRUE;
      ELSE
        queue_prop := FALSE;
      END IF;
        
      rule_info := TO_CHAR(subs.rule);
  
      /* Add subcriber to each physical queue */
      FOR indx IN map_arr.FIRST..map_arr.LAST LOOP
        IF( map_arr(indx).logical_qt != map_arr(indx).physical_qt) THEN
          sys.dbms_aqadm.add_subscriber(
            queue_name => map_arr(indx).physical_queue,
            subscriber => subscriber_info,
            rule => rule_info,
            transformation => subs.transformation,
            queue_to_queue => queue_prop,
            delivery_mode => delivery);
        END IF;
      END LOOP;
    END LOOP;
  EXCEPTION 
    WHEN NO_DATA_FOUND THEN 
      NULL;

    WHEN OTHERS THEN
    IF(subs_cur%ISOPEN) THEN
      CLOSE subs_cur;
    END IF;

    RAISE;
  END replicate_subscribers; 

 ----------------------------------------------------------------------------

  /* internal procedure to create propagation for sharded queues */
  PROCEDURE replicate_propagation(
                qt_schema     IN VARCHAR2, 
                q_name        IN VARCHAR2,
                map_arr       IN map_table_arr) IS

    at_pos           NUMBER;
    dest_q           VARCHAR2(100);
    dblink           VARCHAR2(128);
    
     CURSOR prop_cur IS
       SELECT destination, start_date, propagation_window, latency, 
	      next_time
       FROM all_queue_schedules WHERE schema = qt_schema AND qname = q_name;

  BEGIN
    /* For each propagation of logical queue create a schedule in physical 
						   queue */
    FOR prop IN prop_cur LOOP
      IF(prop.destination = 'AQ$_LOCAL') THEN
	dblink := NULL;
	dest_q := NULL;
      ELSE
	at_pos := INSTRB(prop.destination, '@', 1, 1);
        IF(at_pos = 0) THEN
          dblink := prop.destination;
	  dest_q := NULL;
        ELSE
	  dest_q := NLS_UPPER(SUBSTRB(prop.destination,1,at_pos-1));
	  dblink := NLS_UPPER(SUBSTRB(prop.destination,at_pos+1,
					LENGTHB(prop.destination)-at_pos));
        END IF;
      END IF;
      /* Schedule propagation for all physical queues */
      FOR indx IN map_arr.FIRST..map_arr.LAST LOOP
        IF(map_arr(indx).logical_qt != map_arr(indx).physical_qt) THEN       
          sys.dbms_aqadm.schedule_propagation(
    	     queue_name        => map_arr(indx).physical_queue,
	     destination       => dblink,
	     start_time        => prop.start_date,
	     duration          => prop.propagation_window,
	     next_time         => prop.next_time,
	     latency           => prop.latency,
	     destination_queue =>  dest_q);
        END IF;
      END LOOP;
    END LOOP;

  EXCEPTION
    WHEN OTHERS THEN
    IF(prop_cur%ISOPEN) THEN
      CLOSE prop_cur;
    END IF;
  
  END replicate_propagation;

 ----------------------------------------------------------------------------

  /* procedure to create shards for a queue table. This will replicate all 
     queues and its properties except notification. It will generate code 
     to create dbmsaq package if it doesn't exist.*/

  PROCEDURE make_sharded_queue(qt_name IN VARCHAR2) IS

    map_arr          map_table_arr;
    canon_qt_schema  VARCHAR2(30);
    qtable_name      VARCHAR2(30);
    canon_qt_name    VARCHAR2(30);
    qt_type          VARCHAR2(7);
    obj_type         VARCHAR2(61);
    payload_type     VARCHAR2(61);
    sort_key         VARCHAR2(22);
    sort_list        VARCHAR2(22);
    recpts           VARCHAR2(8);
    mult_cons        BOOLEAN;     
    msg_grping       VARCHAR2(13);
    message_grping   BINARY_INTEGER;
    primary_inst     NUMBER;
    secondary_inst   NUMBER;
    owner_inst       NUMBER;
    usr_comment      VARCHAR2(50);
    secure_ret       VARCHAR2(3);
    secure_q         BOOLEAN;
    compatible       VARCHAR2(6);
    inst_tab         dbms_utility.instance_table;
    inst_cnt         NUMBER;
    enqueue          VARCHAR2(7);
    enq_enabled      BOOLEAN;
    dequeue          VARCHAR2(7);
    deq_enabled      BOOLEAN;
    storage_clause   VARCHAR2(500);
    indx             BINARY_INTEGER := 1;
    found            BOOLEAN;
    unshrd_prog      NUMBER;
    phy_qt_cnt       NUMBER;
    pkg_exist_flg    BOOLEAN;
    typ_exist_flg    BOOLEAN;

    single_inst      EXCEPTION;
    qt_name_null     EXCEPTION;
    unshrd_in_prog   EXCEPTION;
    already_shrded   EXCEPTION; 
    
    CURSOR queues_c(schema IN VARCHAR2, qt_name IN VARCHAR2) IS
        SELECT *
        FROM all_queues 
        WHERE owner = schema AND QUEUE_TABLE = qt_name 
              AND queue_type != 'EXCEPTION_QUEUE';

  BEGIN
    dbms_utility.active_instances(inst_tab, inst_cnt); 
    IF(inst_cnt < 2) THEN   /* If inst_cnt = 0 then its a non-rac setup. */
      RAISE single_inst;    /* If inst_cnt = 1 then rac setup has only */
    END IF;                 /* one active instance --- do not shard */
 
    IF(qt_name IS NULL) THEN
      RAISE qt_name_null;
    END IF;

    parse_name(qt_name, canon_qt_schema, canon_qt_name);
    maptab_access.sel_maptab(canon_qt_schema,
                             canon_qt_name,
                             unshrd_prog,
                             found);
    
    IF(found) THEN   /* Raise error if make sharded queue is called again */
      IF(unshrd_prog = 0) THEN  /* on a already sharded queue table */
         RAISE already_shrded;
      ELSE
        RAISE unshrd_in_prog;
      END IF;
    END IF;
    BEGIN
      select type, object_type, sort_order, recipients, message_grouping,
            compatible, primary_instance, secondary_instance, owner_instance,
            user_comment, secure
      into qt_type, obj_type, sort_key, recpts, msg_grping, compatible, 
          primary_inst, secondary_inst,owner_inst, usr_comment, secure_ret
      from all_queue_tables
      where queue_table = canon_qt_name and owner = canon_qt_schema ;
    EXCEPTION 
      WHEN NO_DATA_FOUND THEN
      RAISE_APPLICATION_ERROR(-20051,' Queue Table '|| canon_qt_schema ||'.' 
                              || canon_qt_name || ' does not exist');
    END;
  
    CASE sort_key
      WHEN 'ENQUEUE_TIME' THEN sort_list := 'ENQ_TIME';
      WHEN 'PRIORITY, ENQUEUE_TIME' THEN sort_list := 'PRIORITY, ENQ_TIME';
      WHEN 'ENQUEUE_TIME, PRIORITY' THEN sort_list := 'ENQ_TIME, PRIORITY';
      ELSE sort_list := sort_key;
    END CASE;

    IF NLS_UPPER(qt_type) = 'RAW' THEN 
      payload_type := qt_type;
    ELSIF NLS_UPPER(qt_type) = 'OBJECT' THEN
      payload_type := obj_type;
    END IF;
     
    IF (recpts = 'MULTIPLE') THEN
      mult_cons := TRUE;
    ELSE
      mult_cons := FALSE;
    END IF;

    IF NLS_UPPER(msg_grping) = 'NONE' THEN
      message_grping := sys.dbms_aqadm.NONE;
    ELSE
      message_grping := sys.dbms_aqadm.TRANSACTIONAL;
    END IF;

    IF NLS_UPPER(secure_ret) = 'YES' THEN
      secure_q := TRUE;
    ELSE
      secure_q := FALSE;
    END IF;
  
    storage_clause := crt_storage_clause(canon_qt_schema, canon_qt_name);  

    /* Create DBMS_AQ package if not already created */
    maptab_access.pkg_exist(pkg_exist_flg);
    maptab_access.type_exist(payload_type, typ_exist_flg);
    
    IF(pkg_exist_flg = FALSE OR typ_exist_flg = FALSE) THEN
      BEGIN
      create_dbmsaq(NULL, NULL);
      EXCEPTION
       WHEN OTHERS THEN
       RAISE_APPLICATION_ERROR(-20051,
        'Insufficient Privileges: Grant execute privilege to wrapperuser on
         all the types accesible to ' || canon_qt_schema);
      END;
    END IF;

    /* Create shards after dbms_aq package creation. IF any error in package
       creation then don't shard the qtable */ 
   /* Create sharded qtable in all instances except instance where original
      queue table exists. */

    FOR i IN 1..inst_cnt LOOP
      IF(inst_tab(i).inst_number = owner_inst) THEN /* If original qtable */
        map_arr(indx).schema := canon_qt_schema;
        map_arr(indx).logical_qt := canon_qt_name;
        map_arr(indx).physical_qt := canon_qt_name;
        map_arr(indx).primary_inst := primary_inst;
        map_arr(indx).secondary_inst := secondary_inst;
        map_arr(indx).unshrd_in_progress := 0;
        indx := indx + 1;
        continue;
      END IF;

      IF i < inst_cnt THEN  --Owner instance is included
        secondary_inst := inst_tab(i+1).inst_number; -- inst_count  
      ELSE
        secondary_inst := inst_tab(1).inst_number; 
      END IF; 

      qtable_name :=  NLS_UPPER('sharded$_qt_'|| 
                                 wrapperuser.qt$_name_seq.nextval);
      sys.dbms_aqadm.create_queue_table(
        queue_table => qtable_name,
        queue_payload_type => payload_type,
        storage_clause => storage_clause,
        sort_list => sort_list,
        multiple_consumers => mult_cons,
        message_grouping => message_grping,
        comment => usr_comment,
        primary_instance => inst_tab(i).inst_number,
        secondary_instance => secondary_inst,
        compatible =>  compatible,
        secure => secure_q );

      map_arr(indx).schema := canon_qt_schema;
      map_arr(indx).logical_qt := canon_qt_name;
      map_arr(indx).physical_qt := qtable_name;
      map_arr(indx).primary_inst := inst_tab(i).inst_number;
      map_arr(indx).secondary_inst := secondary_inst;
      map_arr(indx).unshrd_in_progress := 0;
      indx := indx + 1;
    END LOOP;
    phy_qt_cnt := map_arr.COUNT;

    FOR queues in queues_c(canon_qt_schema, canon_qt_name) LOOP
      shard_queue(canon_qt_schema, canon_qt_name, queues.name, 
                  queues.queue_type, queues.max_retries, queues.retry_delay, 
                  queues.retention, queues.user_comment, map_arr, phy_qt_cnt);
            
      IF(mult_cons = TRUE) THEN
        replicate_subscribers(canon_qt_schema, queues.name, map_arr);  
                                                     /* Add subscribers */
      END IF; 
      replicate_propagation(canon_qt_schema, queues.name, map_arr); 
                                                 /* Replicate propagations */
      IF (TRIM(queues.enqueue_enabled) = 'YES') THEN
        enq_enabled := TRUE;
      ELSE
        enq_enabled := FALSE;
      END IF;

      IF(TRIM(queues.dequeue_enabled) = 'YES')  THEN
        deq_enabled := TRUE;
      ELSE
        deq_enabled := FALSE;
      END IF; 

       /* Start physical queues */
      IF(enq_enabled OR deq_enabled) THEN
        FOR indx IN map_arr.FIRST..map_arr.LAST LOOP
          IF(map_arr(indx).logical_qt != map_arr(indx).physical_qt) THEN
            sys.dbms_aqadm.start_queue(
               queue_name => map_arr(indx).physical_queue,
               enqueue => enq_enabled,
               dequeue => deq_enabled);
          END IF;
        END LOOP; 
      END IF;
    END LOOP; 

    /* Insert into map table */
    maptab_access.arr_ins_maptab(map_arr);

    dbms_output.put_line('Sharding Successful...');
  EXCEPTION
    WHEN single_inst THEN
    RAISE_APPLICATION_ERROR(-20051,
                       'Cannot create sharded queues in single instance');

    WHEN qt_name_null THEN
    RAISE_APPLICATION_ERROR(-20051, 'Queue table name cannot be null');

    WHEN already_shrded THEN
    RAISE_APPLICATION_ERROR(-20051, 'Queue_table:' || NLS_UPPER(qt_name) || 
                                ' is already sharded');
   
    WHEN unshrd_in_prog THEN
    RAISE_APPLICATION_ERROR(-20051, ' Unsharding of ' || NLS_UPPER(qt_name) || 
                             ' in progress');

    WHEN OTHERS THEN
    IF queues_c%ISOPEN THEN
      CLOSE queues_c;
    END IF;

    RAISE;
  END make_sharded_queue;

 --------------------------------------------------------------------------- 
/* This procedure alters the properties of a queue table alongwith changes
   in map table. User must ensure that each instance contains one shard 
   of the queue table. */

  PROCEDURE alter_map_table(
               queue_table          IN     VARCHAR2,
               primary_instance     IN     BINARY_INTEGER DEFAULT NULL,
               secondary_instance   IN     BINARY_INTEGER DEFAULT NULL) IS

    canon_qt_schema     VARCHAR2(30);
    canon_qt_name       VARCHAR2(30);
    new_prim_inst       BINARY_INTEGER;
    new_sec_inst        BINARY_INTEGER;
    old_prim_inst       BINARY_INTEGER;
    old_sec_inst        BINARY_INTEGER;
    sharded             BOOLEAN;
  BEGIN
    parse_name(queue_table, canon_qt_schema, canon_qt_name);
    
    sys.dbms_aqadm.alter_queue_table(
              queue_table => queue_table,
              primary_instance => primary_instance,
              secondary_instance => secondary_instance);

    maptab_access.sel_maptab(canon_qt_schema,
                             canon_qt_name,
                             old_prim_inst,
                             old_sec_inst,
                             sharded);

    new_prim_inst := NVL(primary_instance, old_prim_inst);
    new_sec_inst := NVL(secondary_instance, old_sec_inst);

    IF(primary_instance IS NOT NULL OR secondary_instance IS NOT NULL) THEN
      IF(sharded) THEN
        maptab_access.upd_maptab(canon_qt_schema, canon_qt_name, 
                                  new_prim_inst, new_sec_inst);
      END IF;
    END IF;

  EXCEPTION 
  WHEN OTHERS THEN
  RAISE;
  END alter_map_table;

  ---------------------------------------------------------------------------

  /* this procedure creates queue table. If payload type doesn't exist then 
     recreate dbms_aq_wrapper package */

  PROCEDURE create_queue_table(
              queue_table         IN VARCHAR2,
              queue_payload_type  IN VARCHAR2 ,
              storage_clause      IN VARCHAR2 DEFAULT NULL,
              sort_list           IN VARCHAR2 DEFAULT NULL,
              multiple_consumers  IN BOOLEAN DEFAULT FALSE,
              message_grouping    IN BINARY_INTEGER DEFAULT NONE,
              comment             IN VARCHAR2 DEFAULT NULL,
              primary_instance    IN BINARY_INTEGER DEFAULT 0,
              secondary_instance  IN BINARY_INTEGER DEFAULT 0,
              compatible          IN VARCHAR2 DEFAULT NULL,
              secure              IN BOOLEAN DEFAULT FALSE) IS

    type_exists            INTEGER;
    canon_payload_schema   VARCHAR2(30);
    canon_payload_type     VARCHAR2(30);
    pkg_exist_flg          BOOLEAN;
  BEGIN

     IF (queue_payload_type != 'RAW' AND queue_payload_type != 'raw') THEN  
      parse_name(queue_payload_type, canon_payload_schema, canon_payload_type);
      select count(*) into type_exists from all_queue_tables 
      where object_type = canon_payload_schema||'.'||canon_payload_type
      and type = 'OBJECT' ;
    END IF; 

    IF ((queue_payload_type != 'RAW' AND queue_payload_type != 'raw') 
                                             and type_exists = 0 ) THEN
      /* Dont create dbms_aq_wrapper in every create_queue_table call. 
         If package already exists then re-create it only if payload type
         is new */ 
      maptab_access.pkg_exist(pkg_exist_flg);
      IF(pkg_exist_flg = TRUE) THEN 
        BEGIN /* Create dbms_aq_wrapper package */
          create_dbmsaq(canon_payload_schema, canon_payload_type);
        EXCEPTION
          WHEN OTHERS THEN
          RAISE_APPLICATION_ERROR(-20051,
           'Insufficient Privileges: Grant execute privilege to wrapperuser on '              || queue_payload_type);
        END;
      END IF;
    END IF;
    /* Create table after dbms_aq_wrapper package creation */
    sys.dbms_aqadm.create_queue_table( 
            queue_table => queue_table,
            queue_payload_type => queue_payload_type,
            storage_clause => storage_clause,
            sort_list => sort_list,
            multiple_consumers => multiple_consumers,
            message_grouping => message_grouping,
            comment => comment,
            primary_instance => primary_instance,
            secondary_instance => secondary_instance,
            compatible => compatible,
            secure => secure);

  EXCEPTION
    WHEN OTHERS THEN
     RAISE;
             
  END create_queue_table;

  -----------------------------------------------------------------------------
 
  PROCEDURE alter_queue_table(
               queue_table          IN     VARCHAR2,
               comment              IN     VARCHAR2       DEFAULT NULL,
               primary_instance     IN     BINARY_INTEGER DEFAULT NULL,
               secondary_instance   IN     BINARY_INTEGER DEFAULT NULL) IS
  BEGIN
    sys.dbms_aqadm.alter_queue_table(
              queue_table => queue_table, 
              comment => comment, 
              primary_instance => primary_instance,
              secondary_instance => secondary_instance);
  END alter_queue_table;

 -----------------------------------------------------------------------------
 
  PROCEDURE create_queue(
              queue_name          IN VARCHAR2,
              queue_table         IN VARCHAR2,
              queue_type          IN BINARY_INTEGER DEFAULT NORMAL_QUEUE,
              max_retries         IN NUMBER DEFAULT NULL,
              retry_delay         IN NUMBER DEFAULT 0,
              retention_time      IN NUMBER DEFAULT 0,
              dependency_tracking IN BOOLEAN DEFAULT FALSE,
              comment             IN VARCHAR2 DEFAULT NULL,
              auto_commit         IN BOOLEAN DEFAULT TRUE) IS
 
   canon_qt_schema  VARCHAR2(30);
   canon_q_name     VARCHAR2(30);
   canon_qt_name    VARCHAR2(30);
   canon_q_schema   VARCHAR2(30);
   qt_name          VARCHAR2(65);
   q_name           VARCHAR2(30);
   p_queue          VARCHAR2(30);
   l_queue          VARCHAR2(30);
   queue            VARCHAR2(65);
   p_inst           NUMBER;
   s_inst           NUMBER;
   map_arr          map_table_arr;

  BEGIN
    parse_name(queue_table, canon_qt_schema, canon_qt_name);
    parse_name(queue_name, canon_q_schema, canon_q_name, FALSE);

    maptab_access.qt_arr_sel_maptab(canon_qt_schema, 
                                    canon_qt_name,
                                    map_arr);
    IF(map_arr.count = 0) THEN
      sys.dbms_aqadm.create_queue(
                 queue_name           => queue_name,
                 queue_table          => queue_table,
                 queue_type           => queue_type,
                 max_retries          => max_retries,
                 retry_delay          => retry_delay,
                 retention_time       => retention_time,
                 dependency_tracking  => dependency_tracking,
                 comment              => comment,
                 auto_commit          => auto_commit);
    ELSE
      FOR indx in map_arr.FIRST..map_arr.LAST LOOP
        qt_name := sys.dbms_assert.enquote_name(canon_qt_schema, FALSE) || '.'
              || sys.dbms_assert.enquote_name(map_arr(indx).physical_qt, FALSE);

        IF (map_arr(indx).physical_qt = canon_qt_name) THEN
        q_name := canon_q_name;
        ELSE
        q_name := NLS_UPPER('sharded$_q_' || wrapperuser.q$_name_seq.nextval);
        END IF;

        IF(canon_q_schema IS NOT NULL) THEN
          queue := sys.dbms_assert.enquote_name(canon_q_schema, FALSE) || '.' 
                  || sys.dbms_assert.enquote_name(q_name, FALSE);
        ELSE
          queue := sys.dbms_assert.enquote_name(q_name, FALSE);
        END IF;
        sys.dbms_aqadm.create_queue(
                  queue_name           => queue,
                  queue_table          => qt_name,
                  queue_type           => queue_type,
                  max_retries          => max_retries,
                  retry_delay          => retry_delay,
                  retention_time       => retention_time,
                  dependency_tracking  => dependency_tracking,
                  comment              => comment,
                  auto_commit          => auto_commit);

        IF (map_arr(indx).physical_queue IS NOT NULL)THEN
           maptab_access.ins_maptab( canon_qt_schema, canon_qt_name,
                    map_arr(indx).physical_qt, canon_q_name, q_name, 
                    map_arr(indx).primary_inst, map_arr(indx).secondary_inst, 
                    0);
        ELSE
          maptab_access.upd_maptab( canon_qt_schema,
                                    canon_q_name,
                                    q_name,
                                    canon_qt_name, 
                                    map_arr(indx).physical_qt);
        END IF;

      END LOOP;
    END IF;
  EXCEPTION 
    WHEN OTHERS THEN
    RAISE;
  END create_queue;

 ----------------------------------------------------------------------------
  -- INPUT PARAMETERS
  --   q_name       - name of the queue
  --   auto_commit  - call is commited if this is true

  PROCEDURE drop_queue(
                   queue_name    IN VARCHAR2,
                   auto_commit   IN  BOOLEAN DEFAULT TRUE) IS

    canon_q_schema       VARCHAR2(30);
    canon_q_name         VARCHAR2(30);
    q_name               VARCHAR2(65);
    map_arr              map_table_arr;
 
  BEGIN
    parse_name(queue_name, canon_q_schema, canon_q_name);
    
    maptab_access.q_arr_sel_maptab(canon_q_schema,
                                    canon_q_name,
                                    map_arr);
  
     /* Queue is not sharded */
    IF(map_arr.count = 0) THEN
      sys.dbms_aqadm.drop_queue(
                   queue_name => queue_name,
                   auto_commit => auto_commit); 
    ELSE 
      FOR indx in map_arr.FIRST..map_arr.LAST LOOP
        q_name := sys.dbms_assert.enquote_name(canon_q_schema, FALSE) || '.'
               || sys.dbms_assert.enquote_name(map_arr(indx).physical_queue, FALSE);

        sys.dbms_aqadm.drop_queue(
                   queue_name => q_name,
                   auto_commit => auto_commit);

        maptab_access.q_del_maptab(canon_q_schema,
                                    map_arr(indx).physical_queue);
      END LOOP;
    END IF;

  EXCEPTION
    WHEN OTHERS THEN    
    RAISE; 
  END drop_queue;

  --------------------------------------------------------------------------

  PROCEDURE drop_queue_table(
                        queue_table  IN     VARCHAR2,
                        force        IN     BOOLEAN DEFAULT FALSE,
                        auto_commit  IN     BOOLEAN DEFAULT TRUE) IS

    canon_qt_schema    VARCHAR2(30);
    canon_qt_name      VARCHAR2(30);
    qt_name            VARCHAR2(65);
    phyqt_arr          qt_list;

  BEGIN
    parse_name(queue_table, canon_qt_schema, canon_qt_name);
    maptab_access.qt_arr_sel_maptab(canon_qt_schema,
                                    canon_qt_name,
                                    phyqt_arr);

    sys.dbms_aqadm.drop_queue_table(
                 queue_table  => queue_table,
                 force        => force,
                 auto_commit  => auto_commit);
    IF(phyqt_arr.COUNT > 0) THEN
      maptab_access.qt_del_maptab(canon_qt_schema,canon_qt_name);
      FOR indx IN phyqt_arr.FIRST..phyqt_arr.LAST LOOP
       IF(canon_qt_name != phyqt_arr(indx)) THEN  
        qt_name := sys.dbms_assert.enquote_name(canon_qt_schema, FALSE) || '.'
                || sys.dbms_assert.enquote_name(phyqt_arr(indx), FALSE);

        sys.dbms_aqadm.drop_queue_table(
                   queue_table  => qt_name,
                   force        => force,
                   auto_commit  => auto_commit);
        maptab_access.qt_del_maptab(canon_qt_schema, 
                                          phyqt_arr(indx));
        END IF;
      END LOOP;
                                     
    END IF;

  EXCEPTION
    WHEN OTHERS THEN
    RAISE;
  END drop_queue_table;

  --------------------------------------------------------------------------

  PROCEDURE start_queue(
                  queue_name   IN   VARCHAR2,
                  enqueue      IN   BOOLEAN DEFAULT TRUE,
                  dequeue      IN   BOOLEAN DEFAULT TRUE) IS

    canon_q_schema    VARCHAR2(30);
    canon_q_name      VARCHAR2(30);
    q_name            VARCHAR2(65);
    map_arr           map_table_arr;

  BEGIN
    parse_name(queue_name, canon_q_schema, canon_q_name);
    maptab_access.q_arr_sel_maptab(canon_q_schema,
                                    canon_q_name, map_arr);

     /* Queue is not sharded */
    IF(map_arr.count = 0) THEN
      sys.dbms_aqadm.start_queue(
                   queue_name => queue_name,
                   enqueue => enqueue,
                   dequeue => dequeue);
    ELSE
      FOR indx in map_arr.FIRST..map_arr.LAST LOOP
        q_name := sys.dbms_assert.enquote_name(canon_q_schema, FALSE) || '.'
             || sys.dbms_assert.enquote_name(map_arr(indx).physical_queue, FALSE);
 
        sys.dbms_aqadm.start_queue(
                   queue_name => q_name,
                   enqueue => enqueue,
                   dequeue => dequeue);
      END LOOP;
    END IF;
  
  EXCEPTION 
    WHEN OTHERS THEN
    RAISE; 
  END start_queue;

 -----------------------------------------------------------------------------

  PROCEDURE stop_queue(
                 queue_name    IN   VARCHAR2,
                 enqueue       IN   BOOLEAN DEFAULT TRUE,
                 dequeue       IN   BOOLEAN DEFAULT TRUE,
                 wait          IN   BOOLEAN DEFAULT TRUE) IS

    canon_q_schema   VARCHAR2(30);
    canon_q_name      VARCHAR2(30);
    q_name            VARCHAR2(65);
    map_arr            map_table_arr;   
 
  BEGIN
    parse_name(queue_name, canon_q_schema, canon_q_name);
    maptab_access.q_arr_sel_maptab(canon_q_schema,
                                    canon_q_name,
                                    map_arr);


     /* Queue is not sharded */
    IF(map_arr.count = 0) THEN
      sys.dbms_aqadm.stop_queue(
                     queue_name => queue_name,
                     enqueue => enqueue,
                     dequeue => dequeue,
                     wait => wait);
    ELSE
      FOR indx in map_arr.FIRST..map_arr.LAST LOOP
        q_name := sys.dbms_assert.enquote_name(canon_q_schema, FALSE) || '.'
           || sys.dbms_assert.enquote_name(map_arr(indx).physical_queue, FALSE);

        sys.dbms_aqadm.stop_queue(
                     queue_name => q_name,
                     enqueue => enqueue,
                     dequeue => dequeue,
                     wait => wait);

      END LOOP;
    END IF;

  EXCEPTION 
    WHEN OTHERS THEN
    RAISE;
  END stop_queue;

 -----------------------------------------------------------------------------

  PROCEDURE alter_queue(
                 queue_name       IN     VARCHAR2,
                 max_retries      IN     NUMBER   DEFAULT NULL,
                 retry_delay      IN     NUMBER   DEFAULT NULL,
                 retention_time   IN     NUMBER   DEFAULT NULL,
                 auto_commit      IN     BOOLEAN  DEFAULT TRUE,
                 comment          IN     VARCHAR2 DEFAULT NULL) IS

    canon_q_schema    VARCHAR2(30);
    canon_q_name      VARCHAR2(30);
    q_name            VARCHAR2(65);
    map_arr           map_table_arr;

  BEGIN
    parse_name(queue_name, canon_q_schema, canon_q_name);
    maptab_access.q_arr_sel_maptab(canon_q_schema,
                                    canon_q_name,
                                    map_arr);

    /* Queue is not sharded */
    IF(map_arr.count = 0) THEN
      sys.dbms_aqadm.alter_queue(
                     queue_name => queue_name,
                     max_retries => max_retries,
                     retry_delay => retry_delay,
                     retention_time => retention_time,
                     auto_commit => auto_commit,
                     comment => comment);
    ELSE
      FOR indx in map_arr.FIRST..map_arr.LAST LOOP
        q_name := sys.dbms_assert.enquote_name(canon_q_schema, FALSE) || '.'|| 
              sys.dbms_assert.enquote_name(map_arr(indx).physical_queue, FALSE);

        sys.dbms_aqadm.alter_queue(
                     queue_name => q_name,
                     max_retries => max_retries,
                     retry_delay => retry_delay,
                     retention_time => retention_time,
                     auto_commit => auto_commit,
                     comment => comment);
      END LOOP;
    END IF;

  EXCEPTION
    WHEN OTHERS THEN
    RAISE;
  END alter_queue;
  
 ----------------------------------------------------------------------------                                 
  PROCEDURE add_subscriber(
                queue_name        IN     VARCHAR2,
                subscriber        IN     SYS.AQ$_AGENT,
                rule              IN     VARCHAR2 DEFAULT NULL,
                transformation    IN     VARCHAR2 DEFAULT NULL,
                queue_to_queue    IN     BOOLEAN  DEFAULT FALSE,
                delivery_mode     IN     PLS_INTEGER DEFAULT PERSISTENT) IS

    canon_q_schema    VARCHAR2(30);
    canon_q_name      VARCHAR2(30);
    q_name            VARCHAR2(65);
    map_arr           map_table_arr;

  BEGIN
    parse_name(queue_name, canon_q_schema, canon_q_name);
    maptab_access.q_arr_sel_maptab(canon_q_schema,
                                    canon_q_name,
                                    map_arr);

    /* Queue is not sharded */
    IF(map_arr.count = 0) THEN
      sys.dbms_aqadm.add_subscriber(
                     queue_name => queue_name,
                     subscriber => subscriber,
                     rule => rule,
                     transformation => transformation,
                     queue_to_queue => queue_to_queue,
                     delivery_mode => delivery_mode);
    ELSE
      FOR indx IN map_arr.FIRST..map_arr.LAST LOOP
        q_name := sys.dbms_assert.enquote_name(canon_q_schema, FALSE) || '.' ||
              sys.dbms_assert.enquote_name(map_arr(indx).physical_queue, FALSE);
    
        sys.dbms_aqadm.add_subscriber(
                     queue_name => q_name,
                     subscriber => subscriber,
                     rule => rule,
                     transformation => transformation,
                     queue_to_queue => queue_to_queue,
                     delivery_mode => delivery_mode);

      END LOOP;
    END IF;

  EXCEPTION
    WHEN OTHERS THEN
    RAISE;
  END add_subscriber;

 ----------------------------------------------------------------------------

  PROCEDURE alter_subscriber(
                  queue_name       IN     VARCHAR2,
                  subscriber       IN     SYS.AQ$_AGENT,
                  rule             IN     VARCHAR2,
                  transformation   IN     VARCHAR2) IS

    canon_q_schema    VARCHAR2(30);
    canon_q_name      VARCHAR2(30);
    q_name            VARCHAR2(65);
    map_arr           map_table_arr;

  BEGIN
    parse_name(queue_name, canon_q_schema, canon_q_name);
    maptab_access.q_arr_sel_maptab(canon_q_schema,
                                    canon_q_name,
                                    map_arr);

    /* Queue is not sharded */
    IF(map_arr.count = 0) THEN
      sys.dbms_aqadm.alter_subscriber(
                    queue_name => queue_name,
                    subscriber => subscriber,
                    rule => rule,
                    transformation => transformation);
    ELSE
      FOR indx in map_arr.FIRST..map_arr.LAST LOOP
        q_name := sys.dbms_assert.enquote_name(canon_q_schema, FALSE) || '.' ||
              sys.dbms_assert.enquote_name(map_arr(indx).physical_queue, FALSE);

        sys.dbms_aqadm.alter_subscriber(
                    queue_name => q_name,
                    subscriber => subscriber,
                    rule => rule,
                    transformation => transformation);

      END LOOP;
    END IF;
  
  EXCEPTION
    WHEN OTHERS THEN
    RAISE;
  END alter_subscriber;

 ----------------------------------------------------------------------------

  PROCEDURE recover_sharded_queue(qt_name IN VARCHAR2) IS
  
    canon_qt_name     VARCHAR2(30);
    canon_qt_schema   VARCHAR2(30);
    queue_table       VARCHAR2(65);
    phyqt_arr         qt_list;

  BEGIN
    /* Drop all physical queue tables and delete all map table entires */
    parse_name(qt_name, canon_qt_schema, canon_qt_name);
    maptab_access.qt_arr_sel_maptab(canon_qt_schema,
                                    canon_qt_name,
                                    phyqt_arr);

    FOR indx in phyqt_arr.FIRST..phyqt_arr.LAST LOOP
      queue_table := sys.dbms_assert.enquote_name(canon_qt_schema, FALSE) || '.'
           || sys.dbms_assert.enquote_name(phyqt_arr(indx), FALSE);

      sys.dbms_aqadm.drop_queue_table(
          queue_table => queue_table,
          force => TRUE);
      maptab_access.qt_del_maptab(canon_qt_schema, 
                                        phyqt_arr(indx));
    END LOOP;

  EXCEPTION
    WHEN OTHERS THEN
    RAISE;
  END recover_sharded_queue;

 ----------------------------------------------------------------------------

 PROCEDURE remove_subscriber(
                  queue_name     IN     VARCHAR2,
                  subscriber     IN     SYS.AQ$_AGENT) IS
 
    canon_q_schema   VARCHAR2(30);
    canon_q_name     VARCHAR2(30);
    q_name           VARCHAR2(65);  
    map_arr          map_table_arr;

  BEGIN
    parse_name(queue_name, canon_q_schema, canon_q_name);
    maptab_access.q_arr_sel_maptab(canon_q_schema,
                                    canon_q_name,
                                    map_arr);


    /* Queue is not sharded */
    IF(map_arr.count = 0) THEN
      sys.dbms_aqadm.remove_subscriber(
                    queue_name => queue_name,
                    subscriber => subscriber);
    ELSE
      FOR indx in map_arr.FIRST..map_arr.LAST LOOP
         q_name := sys.dbms_assert.enquote_name(canon_q_schema, FALSE) || '.'|| 
              sys.dbms_assert.enquote_name(map_arr(indx).physical_queue, FALSE);

        sys.dbms_aqadm.remove_subscriber(
                    queue_name => q_name,
                    subscriber => subscriber);

      END LOOP;
    END IF;

  EXCEPTION
    WHEN OTHERS THEN
    RAISE;
  END remove_subscriber;

 ---------------------------------------------------------------------------

  FUNCTION queue_subscribers(
                 queue_name    IN  VARCHAR2)
      RETURN aq$_subscriber_list_t AS

    v_row       BINARY_INTEGER;
    subs_list   sys.dbms_aqadm.aq$_subscriber_list_t;
    ret_list    aq$_subscriber_list_t; 

  BEGIN
    subs_list := sys.dbms_aqadm.queue_subscribers(queue_name => queue_name);
    /* Convert sys.dbms_aqadm.<data_type> to dbms_aqadm.<data_type> */
    v_row := subs_list.FIRST;
    LOOP
     EXIT WHEN v_row IS NULL;
     ret_list(v_row) := subs_list(v_row);
     v_row := subs_list.NEXT(v_row);
    END LOOP;
    RETURN ret_list;  

  END queue_subscribers;

 ---------------------------------------------------------------------------
  -- This procedure grants access privilege on a queue object to a user
  -- or a role.  The access privilege can be granted with grant option.
  -- Currently the allowable privilages are 'ENQUEUE' and 'DEQUEUE'.
  --
  -- INPUT PARAMETERS
  --   privilege       - 'ENQUEUE', 'DEQUEUE' or 'ALL'
  --   queue_name      - name of the queue
  --   grantee         - user or role name of grantees
  --   grant_option    - TRUE if privilege should be granted with GRANT

  PROCEDURE grant_queue_privilege(
                    privilege      IN      VARCHAR2,
                    queue_name     IN      VARCHAR2,
                    grantee        IN      VARCHAR2,
                    grant_option   IN      BOOLEAN := FALSE) IS

    canon_q_schema   VARCHAR2(30);
    canon_q_name     VARCHAR2(30);
    q_name           VARCHAR2(65);
    map_arr          map_table_arr;  

  BEGIN
    parse_name(queue_name, canon_q_schema, canon_q_name);
    maptab_access.q_arr_sel_maptab(canon_q_schema,
                                    canon_q_name,
                                    map_arr);

    /* Queue is not sharded */
    IF(map_arr.count = 0) THEN
      sys.dbms_aqadm.grant_queue_privilege(
             privilege    => privilege,
             queue_name  =>  queue_name,
             grantee      => grantee,
             grant_option => grant_option);
    ELSE
      FOR indx in map_arr.FIRST..map_arr.LAST LOOP
        q_name := sys.dbms_assert.enquote_name(canon_q_schema, FALSE) || '.' || 
              sys.dbms_assert.enquote_name(map_arr(indx).physical_queue, FALSE);

        sys.dbms_aqadm.grant_queue_privilege(
             privilege    => privilege,
             queue_name  =>  q_name,
             grantee      => grantee,
             grant_option => grant_option);

      END LOOP;
    END IF;

  EXCEPTION
    WHEN OTHERS THEN
    RAISE;
  END grant_queue_privilege;

 --------------------------------------------------------------------------
  -- This procedure grants AQ system privilege to a user or a role.
  -- The system  privilege can be granted with admin option.
  --
  -- INPUT PARAMETERS
  --   privilege       - 'ENQUEUE_ANY', 'DEQUEUE_ANY' or 'MANAGE_ANY'
  --   grantee         - user or role name of grantees
  --   admin_option    - TRUE if privilege should be granted with admin
  --                     option
  PROCEDURE grant_system_privilege(
                  privilege       IN      VARCHAR2,
                  grantee         IN      VARCHAR2,
                  admin_option    IN      BOOLEAN := FALSE) IS
    
  BEGIN
    sys.dbms_aqadm.grant_system_privilege(
           privilege    => privilege,
           grantee      => grantee,    
           admin_option => admin_option);
  
  END grant_system_privilege;
  
 ---------------------------------------------------------------------------
  -- This procedure revokes AQ object privilege from a user or a role.
  -- The object privilege granted by the revokee through the grant option
  -- will also be revoked.
  --
  -- INPUT PARAMETERS
  --   privilege       - 'ENQUEUE', 'DEQUEUE' or 'ALL'
  --   queue_name      - name of the queue
  --   grantee         - user or role name of grantees
  PROCEDURE revoke_queue_privilege(
        privilege               IN      VARCHAR2,
        queue_name              IN      VARCHAR2,
        grantee                 IN      VARCHAR2) IS

    canon_q_schema   VARCHAR2(30);
    canon_q_name     VARCHAR2(30);
    q_name           VARCHAR2(65);
     map_arr         map_table_arr;

  BEGIN
    parse_name(queue_name, canon_q_schema, canon_q_name);
    maptab_access.q_arr_sel_maptab(canon_q_schema,
                                    canon_q_name,
                                    map_arr);

    /* Queue is not sharded */
    IF(map_arr.count = 0) THEN
     sys.dbms_aqadm.revoke_queue_privilege(
         privilege  => privilege,
         queue_name => queue_name,
         grantee    => grantee);
    ELSE
      FOR indx in map_arr.FIRST..map_arr.LAST LOOP
        q_name := sys.dbms_assert.enquote_name(canon_q_schema, FALSE) || '.' ||
              sys.dbms_assert.enquote_name(map_arr(indx).physical_queue, FALSE);

        sys.dbms_aqadm.revoke_queue_privilege(
            privilege  => privilege,
            queue_name => q_name,
            grantee    => grantee);

      END LOOP;
    END IF;

  EXCEPTION 
    WHEN OTHERS THEN
    RAISE;   
  END revoke_queue_privilege;

 --------------------------------------------------------------------------
  -- This procedure revokes AQ system privilege from a user or a role.
  -- The system privilege granted by the revokee through the admin option
  -- will not be revoked.
  --
  -- INPUT PARAMETERS
  --   privilege       - 'ENQUEUE_ANY', 'DEQUEUE_ANY' or 'MANAGE_ANY'
  --   grantee         - user or role name of grantees

 PROCEDURE revoke_system_privilege(
                  privilege   IN      VARCHAR2,
                  grantee     IN      VARCHAR2) IS
  BEGIN
    sys.dbms_aqadm.revoke_system_privilege(
        privilege => privilege,
        grantee   => grantee);
    
  END revoke_system_privilege;

 ----------------------------------------------------------------------------
-- This procedure will return the toid, version number and tds of the message
-- type for the specified queue.  The toid will be null and the other values 
-- undefined if the message type is RAW.  If the tdsind is FALSE then the 
-- tds value is not returned.

 PROCEDURE get_type_info(
        schema                      IN     VARCHAR2,
        qname                       IN     VARCHAR2,
        gettds                      IN     BOOLEAN,
        rc                          OUT    BINARY_INTEGER,
        toid                        OUT    RAW,
        version                     OUT    NUMBER,
        tds                         OUT    LONG RAW,
        queue_style                 OUT    VARCHAR2,
        network_name                OUT    VARCHAR2) IS

  BEGIN
    sys.dbms_aqadm.get_type_info(
        schema  => schema,  
        qname   => qname, 
        gettds  => gettds,
        rc      => rc, 
        toid    => toid,
        version => version,
        tds     => tds,
        queue_style  => queue_style,
        network_name => network_name);

  END get_type_info;

-------------------------------------------------------------------------------
-- This procedure will return the toid, version number and tds of the message
-- type for the specified queue.  The toid will be null and the other values 
-- undefined if the message type is RAW.  If the tdsind is FALSE then the 
-- tds value is not returned.

  PROCEDURE get_type_info(
        schema                      IN     VARCHAR2,
        qname                       IN     VARCHAR2,
        gettds                      IN     BOOLEAN,
        rc                          OUT    BINARY_INTEGER,
        toid                        OUT    RAW,
        version                     OUT    NUMBER,
        tds                         OUT    LONG RAW) IS

  BEGIN
    sys.dbms_aqadm.get_type_info(
        schema  => schema,
        qname   => qname,
        gettds  => gettds,
        rc      => rc,
        toid    => toid,
        version => version,
        tds     => tds);

  END get_type_info;
  
 -----------------------------------------------------------------------------
 -- This function will compare the types for the two queues and return 1 if 
 -- they are equal, 0 if they are not.
 
  PROCEDURE verify_queue_types (
                src_queue_name  IN      VARCHAR2,
                dest_queue_name IN      VARCHAR2,
                destination     IN      VARCHAR2,
                rc              OUT     BINARY_INTEGER,
                transformation  IN      VARCHAR2 DEFAULT NULL) IS 
  
  BEGIN
     sys.dbms_aqadm.verify_queue_types (
             src_queue_name  => src_queue_name,
             dest_queue_name => dest_queue_name,
             destination     => destination,
             rc              => rc,
             transformation  => transformation);
  END verify_queue_types;

   ---------------------------------------------------------------------------
  PROCEDURE get_prop_seqno(qid    IN    BINARY_INTEGER,
                           dqname IN    VARCHAR2,
                           dbname IN    VARCHAR2,
                           seq    OUT   BINARY_INTEGER) IS
  BEGIN

    sys.dbms_aqadm.get_prop_seqno(
             qid    => qid,
             dqname => dqname,
             dbname => dbname,
             seq    => seq);

  END get_prop_seqno;

 ---------------------------------------------------------------------------
-- This function will compare the types for the two queues and return 1 if 
-- they are equal, 0 if they are not. It is for use by propagation.  

  PROCEDURE verify_queue_types_no_queue (
                src_queue_name  IN      VARCHAR2,
                dest_queue_name IN      VARCHAR2,
                destination     IN      VARCHAR2,
                rc              OUT     BINARY_INTEGER,
                transformation  IN      VARCHAR2 DEFAULT NULL) IS

   BEGIN
    sys.dbms_aqadm.verify_queue_types_no_queue (  
                src_queue_name  => src_queue_name,
                dest_queue_name => dest_queue_name,
                destination     => destination,
                transformation  => transformation,
                rc              => rc);

  END verify_queue_types_no_queue;

  ---------------------------------------------------------------------------

-- This function will compare the types for the two queues and return 1 if 
-- they are equal, 0 if they are not. It is for use by propagation.  

  PROCEDURE verify_queue_types_get_nrp(
        src_queue_name  IN      VARCHAR2,
        dest_queue_name IN      VARCHAR2,
        destination     IN      VARCHAR2 DEFAULT NULL,
        rc              OUT     BINARY_INTEGER,
        transformation  IN      VARCHAR2 DEFAULT NULL) IS

  BEGIN
    sys.dbms_aqadm.verify_queue_types_get_nrp (
                src_queue_name  => src_queue_name,
                dest_queue_name => dest_queue_name,
                destination     => destination,
                transformation  => transformation,
                rc              => rc);

  END verify_queue_types_get_nrp;

  ---------------------------------------------------------------------------
  -- This procedure recovers propagation from a source queue to a given 
  -- destination.

  PROCEDURE recover_propagation(
        schema          IN      VARCHAR2,
        queue_name      IN      VARCHAR2,
        destination     IN      VARCHAR2,
        protocol        IN      BINARY_INTEGER DEFAULT TTC,
        url             IN      VARCHAR2 DEFAULT NULL,
        username        IN      VARCHAR2 DEFAULT NULL,
        passwd          IN      VARCHAR2 DEFAULT NULL,
        trace           IN      BINARY_INTEGER DEFAULT 0,
        destq           IN      BINARY_INTEGER default 0) IS
  
  BEGIN
    sys.dbms_aqadm.recover_propagation(schema, queue_name, destination,
                                          protocol, url, username, passwd,
                                          trace, destq);
  END recover_propagation;

 ----------------------------------------------------------------------------
  
  -- This procedure schedules propagation from a queue to a specific
  -- destination.  
  --
  -- INPUT PARAMETERS
  --   queue_name     - name of the queue to be scheduled    
  --   destination    - address of the destination.  Only address format 
  --                    supported is "@DBLINK SYNONYM".  The synonym "LOOP"
  --                    is used to propagate to other local queues
  --   start_time     - date at which to start propagation             
  --   duration       - propagation window in seconds. A NULL value means 
  --                    the propagation window is until explicitly stopped.
  --   next_time      - date function to compute the start of next propagation
  --                    window from the end of the previous window.  If this
  --                    value is NULL, propagation will be stopped at the 
  --                    end of the current window
  --   latency        - maximum wait, in seconds, after a message is enqueued 
  --                    before it is propagated (in the propagation window)
  --

  PROCEDURE schedule_propagation(
        queue_name                  IN     VARCHAR2,
        destination                 IN     VARCHAR2,
        start_time                  IN     TIMESTAMP WITH TIME ZONE,
        duration                    IN     NUMBER,
        next_time                   IN     VARCHAR2,
        latency                     IN     NUMBER,
        destination_queue           IN     VARCHAR2) IS

    canon_q_schema   VARCHAR2(30);
    canon_q_name     VARCHAR2(30);
    q_name           VARCHAR2(65);
    map_arr           map_table_arr;

  BEGIN
    parse_name(queue_name, canon_q_schema, canon_q_name);
    maptab_access.q_arr_sel_maptab(canon_q_schema,
                                    canon_q_name,
                                    map_arr);

    /* Queue is not sharded */
    IF(map_arr.count = 0) THEN
      sys.dbms_aqadm.schedule_propagation(
                  queue_name => queue_name,
                  destination => destination,
                  start_time => start_time,
                  duration => duration,
                  next_time => next_time,
                  latency => latency,
                  destination_queue => destination_queue); 
    ELSE
      FOR indx in map_arr.FIRST..map_arr.LAST LOOP
        q_name := sys.dbms_assert.enquote_name(canon_q_schema, FALSE) || '.' ||
             sys.dbms_assert.enquote_name(map_arr(indx).physical_queue, FALSE);

        sys.dbms_aqadm.schedule_propagation(
                  queue_name => q_name,
                  destination => destination,
                  start_time => start_time,
                  duration => duration,
                  next_time => next_time,
                  latency => latency,
                  destination_queue => destination_queue);
      END LOOP;
    END IF;
  
  EXCEPTION
    WHEN OTHERS THEN
    RAISE;
  END schedule_propagation;

 -----------------------------------------------------------------------------
  
  -- This procedure is used to unschedule an existing propagation
  -- schedule. 

  PROCEDURE unschedule_propagation(
                      queue_name           IN     VARCHAR2,
                      destination          IN     VARCHAR2,
                      destination_queue    IN     VARCHAR2) IS
  
    canon_q_schema   VARCHAR2(30);
    canon_q_name     VARCHAR2(30);
    q_name           VARCHAR2(65);
    map_arr          map_table_arr;

  BEGIN
    parse_name(queue_name, canon_q_schema, canon_q_name);
    maptab_access.q_arr_sel_maptab(canon_q_schema,
                                    canon_q_name,
                                    map_arr);
    /* Queue is not sharded */
    IF(map_arr.count = 0) THEN
      sys.dbms_aqadm.unschedule_propagation(
                    queue_name => queue_name,
                    destination => destination,
                    destination_queue => destination_queue);
    ELSE
      FOR indx in map_arr.FIRST..map_arr.LAST LOOP
        q_name := sys.dbms_assert.enquote_name(canon_q_schema, FALSE) || '.' ||
              sys.dbms_assert.enquote_name(map_arr(indx).physical_queue, FALSE);

        sys.dbms_aqadm.unschedule_propagation(
                    queue_name => q_name,
                    destination => destination,
                    destination_queue => destination_queue);
      END LOOP;
    END IF;
 
  EXCEPTION
    WHEN OTHERS THEN
    RAISE;
   
  END unschedule_propagation;

 -----------------------------------------------------------------------------
  -- This procedure is used to alter the parameters of an existing propagation
  -- schedule. 
  --
  -- INPUT PARAMETERS
  --   queue_name     - name of the queue to be scheduled    
  --   destination    - address of the destination.  Only address format 
  --                    supported is "@DBLINK SYNONYM". 
  --   duration       - propagation window in seconds. A NULL value means 
  --                    the propagation window is until explicitly stopped.
  --   next_time      - date function to compute the start of next propagation
  --                    window from the end of the previous window.  If this
  --                    value is NULL, propagation will be stopped at the 
  --                    end of the current window
  --   latency        - maximum wait, in seconds, after a message is enqueued 
  --                    before it is propagated (in the propagation window)
  --
  
  PROCEDURE alter_propagation_schedule(
        queue_name                  IN     VARCHAR2,
        destination                 IN     VARCHAR2 DEFAULT NULL,
        duration                    IN     NUMBER DEFAULT NULL,
        next_time                   IN     VARCHAR2 DEFAULT NULL,
        latency                     IN     NUMBER DEFAULT 60,
        destination_queue           IN     VARCHAR2 DEFAULT NULL) IS

    canon_q_schema   VARCHAR2(30);
    canon_q_name     VARCHAR2(30);
    q_name           VARCHAR2(65);
    map_arr          map_table_arr;

  BEGIN
    parse_name(queue_name, canon_q_schema, canon_q_name);
    maptab_access.q_arr_sel_maptab(canon_q_schema,
                                    canon_q_name,
                                    map_arr);

    /* Queue is not sharded */
    IF(map_arr.count = 0) THEN
      sys.dbms_aqadm.alter_propagation_schedule(
                 queue_name => queue_name,
                 destination => destination,
                 duration => duration,
                 next_time => next_time,
                 latency => latency,
                 destination_queue => destination_queue);
    ELSE
      FOR indx in map_arr.FIRST..map_arr.LAST LOOP
        q_name := sys.dbms_assert.enquote_name(canon_q_schema, FALSE) || '.' ||
              sys.dbms_assert.enquote_name(map_arr(indx).physical_queue, FALSE);

        sys.dbms_aqadm.alter_propagation_schedule(
                 queue_name => q_name,
                 destination => destination,
                 duration => duration,
                 next_time => next_time,
                 latency => latency,
                 destination_queue => destination_queue);
      END LOOP;
    END IF;    
  
  EXCEPTION
    WHEN OTHERS THEN
    RAISE;
  END alter_propagation_schedule;

 -----------------------------------------------------------------------------

  PROCEDURE enable_propagation_schedule(
        queue_name                  IN     VARCHAR2,
        destination                 IN     VARCHAR2 DEFAULT NULL,
        destination_queue           IN     VARCHAR2 DEFAULT NULL) IS

    canon_q_schema    VARCHAR2(30);
    canon_q_name      VARCHAR2(30);
    q_name            VARCHAR2(65);
    map_arr           map_table_arr;

  BEGIN
    parse_name(queue_name, canon_q_schema, canon_q_name);
    maptab_access.q_arr_sel_maptab(canon_q_schema,
                                    canon_q_name,
                                    map_arr);

    /* Queue is not sharded */
    IF(map_arr.count = 0) THEN
      sys.dbms_aqadm.enable_propagation_schedule(
                   queue_name => queue_name,
                   destination => destination,
                   destination_queue => destination_queue);
    ELSE
      FOR indx in map_arr.FIRST..map_arr.LAST LOOP
        q_name := sys.dbms_assert.enquote_name(canon_q_schema, FALSE) || '.' ||
              sys.dbms_assert.enquote_name(map_arr(indx).physical_queue, FALSE);

        sys.dbms_aqadm.enable_propagation_schedule(
                   queue_name => q_name,
                   destination => destination,
                   destination_queue => destination_queue);

      END LOOP;
    END IF;

  EXCEPTION 
    WHEN OTHERS THEN
    RAISE;
  END enable_propagation_schedule;

 -----------------------------------------------------------------------------
  
   PROCEDURE disable_propagation_schedule(
        queue_name                  IN     VARCHAR2,
        destination                 IN     VARCHAR2 DEFAULT NULL,
        destination_queue           IN     VARCHAR2 DEFAULT NULL) IS

    canon_q_schema   VARCHAR2(30);
    canon_q_name     VARCHAR2(30);
    q_name           VARCHAR2(65);
    map_arr          map_table_arr;

  BEGIN
    parse_name(queue_name, canon_q_schema, canon_q_name);
    maptab_access.q_arr_sel_maptab(canon_q_schema,
                                    canon_q_name,
                                    map_arr);

    /* Queue is not sharded */
    IF(map_arr.count = 0) THEN
      sys.dbms_aqadm.disable_propagation_schedule(
                     queue_name => queue_name,
                     destination => destination,
                     destination_queue => destination_queue);
    ELSE
      FOR indx in map_arr.FIRST..map_arr.LAST LOOP
        q_name := sys.dbms_assert.enquote_name(canon_q_schema, FALSE) || '.'|| 
              sys.dbms_assert.enquote_name(map_arr(indx).physical_queue, FALSE);

        sys.dbms_aqadm.disable_propagation_schedule(
                     queue_name => q_name,
                     destination => destination,
                     destination_queue => destination_queue);

      END LOOP;
    END IF;

  EXCEPTION
    WHEN OTHERS THEN
    RAISE;
  END disable_propagation_schedule;

 ----------------------------------------------------------------------------

  PROCEDURE migrate_queue_table(
                 queue_table   IN   VARCHAR2,
                 compatible    IN   VARCHAR2) IS

   canon_qt_schema    VARCHAR2(30);
   canon_qt_name      VARCHAR2(30);
   qt_name            VARCHAR2(65);
   phyqt_arr          qt_list;

  BEGIN
    parse_name(queue_table, canon_qt_schema, canon_qt_name);
    maptab_access.qt_arr_sel_maptab(canon_qt_schema,
                                    canon_qt_name,
                                    phyqt_arr);

    IF(phyqt_arr.count = 0) THEN
       sys.dbms_aqadm.migrate_queue_table(
              queue_table => queue_table,
              compatible => compatible);
    ELSE
      FOR indx in phyqt_arr.FIRST..phyqt_arr.LAST LOOP
       qt_name := sys.dbms_assert.enquote_name(canon_qt_schema, FALSE) || '.' || 
                 sys.dbms_assert.enquote_name(phyqt_arr(indx), FALSE);

        sys.dbms_aqadm.migrate_queue_table(
              queue_table => qt_name,
              compatible => compatible);
    
      END LOOP;
    END IF;
    
  END migrate_queue_table;

 --------------------------------------------------------------------------
  -- This procedure activates the time manager by setting the appropriate
  -- event. The time manger does useful work iff this event is set
  -- PARAMETERS - None

  PROCEDURE start_time_manager IS

  BEGIN
    sys.dbms_aqadm.start_time_manager;
  END start_time_manager;

  -------------------------------------------------------------------------
  -- This procedure activates the time manager by setting the appropriate
  -- event. The time manger does useful work iff this event is set
  -- PARAMETERS - None
  
  PROCEDURE stop_time_manager IS

  BEGIN
     sys.dbms_aqadm.stop_time_manager;
  END stop_time_manager;

 ----------------------------------------------------------------------------

 -- This procedure is used to set the high watermark
 -- INPUT PARAMETERS -
 --   wmvalue - the new value for the watemark - this value is in MB.

  PROCEDURE set_watermark(
               wmvalue    IN       NUMBER) IS
  BEGIN
    sys.dbms_aqadm.set_watermark(  
       wmvalue     => wmvalue);
  END set_watermark;

 -------------------------------------------------------------------------
 -- This procedure is used to get the high watermark
 -- INPUT PARAMETERS -
 --   wmvalue - the set value for the watemark - this value is in MB.

  PROCEDURE get_watermark(
               wmvalue    OUT       NUMBER) IS
  BEGIN
    sys.dbms_aqadm.get_watermark(
       wmvalue     => wmvalue);
  END get_watermark;

 ----------------------------------------------------------------------------
 -- add an alias to LDAP
  PROCEDURE add_alias_to_ldap(
                  alias           IN  VARCHAR2,
                  obj_location    IN  VARCHAR2) IS
  BEGIN
    sys.dbms_aqadm.add_alias_to_ldap(
            alias => alias,
            obj_location => obj_location);
  END add_alias_to_ldap;

 -----------------------------------------------------------------------------
  -- drop an alias from LDAP
  PROCEDURE del_alias_from_ldap(
                      alias    IN  VARCHAR2) IS
  BEGIN
    sys.dbms_aqadm.del_alias_from_ldap(
            alias => alias);
  END del_alias_from_ldap;

  -------------------------------------------------------------------------
  -- This procedure creates a new aq agent. 
  -- This agent can be used to access secure queues.
  --
  -- INPUT PARAMETERS
  -- agent_name            - name of the aq agent
  -- certificate_location  - used authentication when accessing queues 
  --                         via SMTP. Not required for other protocols
  -- enable_http           - whether the user should be given http 
  --                         protocol access
  -- enable_smtp           - whether the user should be given smtp 
  --                         protocol access
  -- enable_anyp           - whether the user should be given access through 
  --                         all protocols (if this is true, all other 
  --                         enable_* parameters aree ignored)

   PROCEDURE create_aq_agent (
                 agent_name                IN VARCHAR2,
                 certificate_location      IN VARCHAR2 DEFAULT NULL,
                 enable_http               IN BOOLEAN DEFAULT FALSE,
                 enable_smtp               IN BOOLEAN DEFAULT FALSE,
                 enable_anyp               IN BOOLEAN DEFAULT FALSE ) IS

  BEGIN
    sys.dbms_aqadm.create_aq_agent(agent_name, certificate_location,
                                   enable_http, enable_smtp, enable_anyp);
  END create_aq_agent;

  -------------------------------------------------------------------------
  -- This procedure alters the properties of an existing aq agent. 
  --
  --
  -- INPUT PARAMETERS
  -- agent_name            - name of the aq agent
  -- certificate_location  - used authentication when accessing queues 
  --                         via SMTP. Not required for other protocols
  -- enable_http           - whether the user should be given http 
  --                         protocol access
  -- enable_smtp           - whether the user should be given smtp 
  --                         protocol access
  -- enable_anyp           - whether the user should be given access through 
  --                         all protocols (if this is true, all other 
  --                         enable_* parameters aree ignored)
  --
  PROCEDURE alter_aq_agent(
                agent_name                IN VARCHAR2,
                certificate_location      IN VARCHAR2 DEFAULT NULL,
                enable_http               IN BOOLEAN DEFAULT FALSE,
                enable_smtp               IN BOOLEAN DEFAULT FALSE,
                enable_anyp               IN BOOLEAN DEFAULT FALSE) IS
    
  BEGIN
    sys.dbms_aqadm.alter_aq_agent(agent_name, certificate_location,
                                  enable_http, enable_smtp, enable_anyp);
  END alter_aq_agent;

 ----------------------------------------------------------------------------
  -- This procedure drops an aq agent entry.  
  --
  -- INPUT PARAMETERS
  -- agent_name        - name of the aq agent to be dropped
  --
  PROCEDURE drop_aq_agent (
                 agent_name   IN VARCHAR2 ) IS
  BEGIN
    sys.dbms_aqadm.drop_aq_agent(agent_name);
  END drop_aq_agent;
 
 ----------------------------------------------------------------------------
  -- This procedure enables the given agent name to have the database
  -- access of the given database user. 
  --
  -- INPUT PARAMETERS
  -- agent_name         - name of the agent to be given access
  -- db_username        - name of the database user 
  --
  PROCEDURE enable_db_access (
            agent_name                IN VARCHAR2,
            db_username               IN VARCHAR2   ) IS

  BEGIN
    sys.dbms_aqadm.enable_db_access(agent_name, db_username);
  END enable_db_access;

  -------------------------------------------------------------------------
  -- This procedure disables the given agent name from having the database
  -- access of the given database user. 
  --
  -- INPUT PARAMETERS
  -- agent_name         - name of the agent to have access revoked
  -- db_username        - name of the database user 
  --
  PROCEDURE disable_db_access (
                  agent_name      IN VARCHAR2,
                  db_username     IN VARCHAR2   ) IS
 
  BEGIN
    sys.dbms_aqadm.disable_db_access(agent_name, db_username);
  END disable_db_access;

 -------------------------------------------------------------------------
  -- enable jms types IN anydata queue tables
  PROCEDURE enable_jms_types(
            queue_table          IN VARCHAR2) IS
 
   canon_qt_schema     VARCHAR2(30);
   canon_qt_name       VARCHAR2(30);
   qt_name             VARCHAR2(65);
   phyqt_arr           qt_list;

  BEGIN
    parse_name(queue_table, canon_qt_schema, canon_qt_name);
    maptab_access.qt_arr_sel_maptab(canon_qt_schema,
                                    canon_qt_name,
                                    phyqt_arr);

    IF(phyqt_arr.count = 0) THEN
      sys.dbms_aqadm.enable_jms_types(queue_table => queue_table);
    ELSE
      FOR indx in phyqt_arr.FIRST..phyqt_arr.LAST LOOP
       qt_name := sys.dbms_assert.enquote_name(canon_qt_schema, FALSE) || '.' ||
                  sys.dbms_assert.enquote_name(phyqt_arr(indx), FALSE);

        sys.dbms_aqadm.enable_jms_types(queue_table => qt_name);
      END LOOP;
    END IF;
 
  EXCEPTION 
    WHEN OTHERS THEN
    RAISE;
  END enable_jms_types; 

 ----------------------------------------------------------------------------
  -- add a connection string to LDAP directory
  PROCEDURE add_connection_to_ldap(
            connection          IN          VARCHAR2,
            host                IN          VARCHAR2,
            port                IN          BINARY_INTEGER,
            sid                 IN          VARCHAR2,
            driver              IN          VARCHAR2 default NULL,
            type                IN          BINARY_INTEGER DEFAULT
                                            AQ_QUEUE_CONNECTION) IS
   BEGIN
    sys.dbms_aqadm.add_connection_to_ldap(
                   connection   =>     connection,
                   host         =>     host,
                   port         =>     port,
                   sid          =>     sid,
                   driver       =>     driver,
                   type         =>     type);
  END;

 ---------------------------------------------------------------------------
   -- add a connection string to LDAP directory
  PROCEDURE add_connection_to_ldap(
            connection          IN          VARCHAR2,
            jdbc_string         IN          VARCHAR2,
            username            IN          VARCHAR2 default NULL,
            password            IN          VARCHAR2 default NULL,
            type                IN          BINARY_INTEGER DEFAULT 
                                            AQ_QUEUE_CONNECTION) IS
  BEGIN
    sys.dbms_aqadm.add_connection_to_ldap(
                   connection   =>      connection,
                   jdbc_string  =>      jdbc_string,
                   username     =>      username,
                   password     =>      password,
                   type         =>      type);
  END;
  
  ---------------------------------------------------------------------------
   -- drop a connection string from LDAP directory
  PROCEDURE del_connection_from_ldap(
            connection          IN          VARCHAR2) IS
  BEGIN
    sys.dbms_aqadm.del_connection_from_ldap(
                   connection   => connection);
  END;

 -----------------------------------------------------------------------------

 -- purge queue table
  PROCEDURE purge_queue_table(
                 queue_table      IN   VARCHAR2,
                 purge_condition  IN   VARCHAR2,
                 purge_options    IN   aq$_purge_options_t) IS

   canon_qt_schema    VARCHAR2(30);
   canon_qt_name      VARCHAR2(30);
   qt_name            VARCHAR2(65);
   purge_opt          sys.dbms_aqadm.aq$_purge_options_t;
   phyqt_arr          qt_list;

  BEGIN
    parse_name(queue_table, canon_qt_schema, canon_qt_name);
    maptab_access.qt_arr_sel_maptab(canon_qt_schema,
                                    canon_qt_name,
                                    phyqt_arr);

    purge_opt.block := purge_options.block;
    purge_opt.delivery_mode := purge_options.delivery_mode;

    IF(phyqt_arr.count = 0) THEN
      sys.dbms_aqadm.purge_queue_table(
                 queue_table     => queue_table,
                 purge_condition => purge_condition,
                 purge_options   => purge_opt);
    ELSE
      FOR indx in phyqt_arr.FIRST..phyqt_arr.LAST LOOP
        qt_name := sys.dbms_assert.enquote_name(canon_qt_schema, FALSE) || 
                   '.' || sys.dbms_assert.enquote_name(phyqt_arr(indx), FALSE);

        sys.dbms_aqadm.purge_queue_table(
                 queue_table     => qt_name,
                 purge_condition => purge_condition,
                 purge_options   => purge_opt);
    
      END LOOP;
    END IF;

  EXCEPTION
    WHEN OTHERS THEN
    RAISE;
  END purge_queue_table;

  ---------------------------------------------------------------------------
  -- This function gets the correlation id. for the last enqueued or
  -- acknowledged message. 
  -- This only works for buffered queues

  PROCEDURE get_replay_info(
        queue_name               IN      VARCHAR2,
        sender_agent             IN      sys.aq$_agent,
        replay_attribute         IN      BINARY_INTEGER,
        correlation              OUT     varchar2) IS
  BEGIN
    sys.dbms_aqadm.get_replay_info(
         queue_name => queue_name,
         sender_agent => sender_agent,
         replay_attribute => replay_attribute,
         correlation => correlation);
  END get_replay_info;

 ----------------------------------------------------------------------------
  -- This function resets the correlation id. for the last enqueued 
  -- or acknowledged message. This only works for buffered queues

  PROCEDURE reset_replay_info(
        queue_name               IN      VARCHAR2,
        sender_agent             IN      sys.aq$_agent,
        replay_attribute         IN      BINARY_INTEGER) IS

  BEGIN
    sys.dbms_aqadm.reset_replay_info(
         queue_name => queue_name,
         sender_agent => sender_agent,
         replay_attribute => replay_attribute);

  END reset_replay_info;

-------------------------------------------------------------------------------

  /* callback procedure for the job created by remove_sharded_queue. This 
    function checks if messages in aq$_queue_table are in processed
    state or queue_table is empty then drop the queue_table */

  PROCEDURE cbk_rm_job(qt_schema IN VARCHAR2, qt_name IN VARCHAR2) IS
    phyqt               varchar2(30);
    cnt                 number;
    state               VARCHAR2(12) := 'PROCESSED';
    phyqt_arr           qt_list;

  BEGIN
    --parse_name(qt_name, canon_qt_schema, canon_qt_name);
    maptab_access.qt_arr_sel_maptab(qt_schema,
                                    qt_name,
                                    phyqt_arr);
    
     /* If no more shards to be dropped, stop the job */
    IF (phyqt_arr.count = 0) THEN
      DBMS_SCHEDULER.DISABLE (
                'AQRWJ$'|| qt_name, force=>true);
      DBMS_SCHEDULER.STOP_JOB (
                'AQRWJ$'|| qt_name, force=>true);

      DBMS_SCHEDULER.DROP_JOB (
               'AQRWJ$'|| qt_name, force=>true);
    END IF;

    FOR indx in phyqt_arr.FIRST..phyqt_arr.LAST LOOP
      EXECUTE IMMEDIATE 'select count(*) from AQ$' ||
                    phyqt_arr(indx) ||
                  ' where msg_state != ''PROCESSED'' ' into cnt;  

     /* queue table has all processed messages or no messages */
      IF (cnt = 0 AND phyqt_arr(indx) != qt_name) THEN

        dbms_aqadm.drop_queue_table(
            queue_table => phyqt_arr(indx),
            force => TRUE);
     
      maptab_access.qt_del_maptab(qt_schema,
                                       phyqt_arr(indx));

      ELSIF (phyqt_arr(indx) = qt_name) THEN
        maptab_access.qt_del_maptab(qt_schema,
                                    phyqt_arr(indx));
      END IF;
    END LOOP;
  
  EXCEPTION
    WHEN OTHERS THEN
    NULL;

  END cbk_rm_job;

-------------------------------------------------------------------------------

  /* internal procedure for cleanup */
  PROCEDURE clean_prg(qt_name IN  VARCHAR2) IS

  BEGIN
    BEGIN 
      dbms_scheduler.disable (
                'AQRWP$' || qt_name, force=>true);
    EXCEPTION 
      WHEN OTHERS THEN NULL; 
    END;

    BEGIN
      dbms_scheduler.drop_program (
                'AQRWP$'|| qt_name, force=>true);
    EXCEPTION 
       WHEN OTHERS THEN 
       NULL; 
     END;

  EXCEPTION
    WHEN OTHERS THEN 
    NULL; 
  END clean_prg;

-------------------------------------------------------------------------------

  PROCEDURE create_prg(qt_name IN varchar2) IS

  BEGIN
    -- Create a new Program in disabled status
    DBMS_SCHEDULER.CREATE_PROGRAM (
        program_name => 'AQRWP$'|| qt_name,
        program_type => 'STORED_PROCEDURE',
        program_action => 'DBMS_AQADM.cbk_rm_job',
        number_of_arguments => 2,
        enabled => FALSE,
        comments => 'Unshards specified queue table');

    -- Create the program's arguments
    DBMS_SCHEDULER.DEFINE_PROGRAM_ARGUMENT (
        program_name => 'AQRWP$'|| qt_name,
        argument_position => 1,
        argument_name => 'qt_schema',
        argument_type => 'VARCHAR2',
        default_value => 'NULL');

    DBMS_SCHEDULER.DEFINE_PROGRAM_ARGUMENT (
        program_name => 'AQRWP$'|| qt_name,
        argument_position => 2,
        argument_name => 'logical_qt_name',
        argument_type => 'VARCHAR2',
        default_value => 'NULL');

        DBMS_SCHEDULER.ENABLE (
         name => 'AQRWP$'|| qt_name);
 dbms_output.put_line('prg created');

  EXCEPTION
    WHEN OTHERS THEN
    NULL; 
  END create_prg;

-------------------------------------------------------------------------------

  PROCEDURE clean_jb(qt_name IN varchar2) IS

  BEGIN
    BEGIN 
      dbms_scheduler.disable (
                'AQRWJ$' || qt_name, force=>true);
    EXCEPTION 
      WHEN OTHERS THEN NULL; 
    END;

    BEGIN
      dbms_scheduler.stop_job (
                'AQRWJ$' || qt_name, force=>true);
    EXCEPTION 
      WHEN OTHERS THEN NULL;
    END;

    BEGIN 
      dbms_scheduler.drop_job (
                'AQRWJ$' || qt_name);
    EXCEPTION 
      WHEN OTHERS THEN 
      NULL; 
    END;
    EXCEPTION 
    WHEN OTHERS THEN 
    NULL; 
    dbms_output.put_line('job cleaned');

  END clean_jb;

-------------------------------------------------------------------------------

  /* creates a job job_<qt_name> which uses a program prg_<qt_name>. This 
    runs periodically and calls procedure cbk_rm_job. This checks if all the 
    the messages have been dequeued and drops the queue table */

  PROCEDURE create_jb(qt_schema IN varchar2,
                      qt_name IN varchar2) is

  BEGIN
    BEGIN
    DBMS_SCHEDULER.CREATE_JOB (
         job_name => 'AQRWJ$'|| qt_name,
         program_name => 'AQRWP$'|| qt_name,
         start_date      => SYSTIMESTAMP,
         repeat_interval => 'freq=minutely; interval=1',
         end_date        => NULL,
         enabled => TRUE ,
         auto_drop => TRUE,
         comments => 'Checks number of messages in queuetable per minute');
     EXCEPTION
       WHEN OTHERS THEN
        RAISE_APPLICATION_ERROR(-20051, 'User does not have create job privilege');
     END;
    DBMS_SCHEDULER.SET_JOB_ARGUMENT_VALUE(
         job_name =>'AQRWJ$'|| qt_name,
         argument_position => 1,
         argument_value => qt_schema);

    DBMS_SCHEDULER.SET_JOB_ARGUMENT_VALUE(
         job_name =>'AQRWJ$'|| qt_name,
         argument_position => 2,
         argument_value => qt_name);
  

   dbms_output.put_line('job created');

   EXCEPTION
     WHEN OTHERS THEN
      RAISE;
  END create_jb;

-------------------------------------------------------------------------------

  /* This procedure submits a job which checks periodically whether all 
     messages in the physical queue_table have been dequeued.If the 
      queue_table is empty then it drops it. */

  PROCEDURE remove_sharded_queue(qt_name IN VARCHAR2) IS

    canon_qt_schema     VARCHAR2(30);
    canon_qt_name       VARCHAR2(30);
    uflg                NUMBER;
    qt_name_null        EXCEPTION;
    unshrd_in_prog      EXCEPTION;
    unsharded_qt        EXCEPTION;
    sharded             BOOLEAN;

  BEGIN
   
    IF(qt_name IS NULL) THEN
      RAISE qt_name_null;
    END IF;

    parse_name(qt_name, canon_qt_schema, canon_qt_name);
    maptab_access.sel_maptab(canon_qt_schema, canon_qt_name, uflg, sharded);

    IF(sharded = FALSE) THEN
      RAISE unsharded_qt;
    END IF;

    IF uflg = 0 THEN

      begin clean_jb (canon_qt_name); end;
      begin clean_prg (canon_qt_name); end;
      begin create_prg (canon_qt_name); end;
      begin create_jb (canon_qt_schema, canon_qt_name); end;
      maptab_access.upd_maptab(canon_qt_schema, canon_qt_name);
      dbms_output.put_line('Remove Shard Started');
    ELSE
      RAISE unshrd_in_prog;
    END IF;
      
  EXCEPTION 
     WHEN qt_name_null THEN
     RAISE_APPLICATION_ERROR(-20051, 'Queue table name cannot be null');

     WHEN unsharded_qt THEN
     RAISE_APPLICATION_ERROR(-20051, 'Queue table:' || NLS_UPPER(qt_name) || 
                            ' is not sharded');

      WHEN unshrd_in_prog THEN
      RAISE_APPLICATION_ERROR(-20051, ' Remove shard of '
                       || NLS_UPPER(qt_name) ||' already started');

     WHEN OTHERS THEN
     RAISE;

 END remove_sharded_queue;

END dbms_aqadm_wrapper;
/

show errors


